////////////////////////////////////////////////
//
//     -- Mikael Beckius 2009 --
//
////////////////////////////////////////////////

#include <reent.h>
#include <eikenv.h>
#include <e32cmn.h>
#include <string.h>

#include "fighter512.h"
#include "S60Gles2View.h"
#include "vertexShader.h"
#include "pixelShader.h"

#include <S60Gles2.rsg>

#ifndef GL_BGRA
#define GL_BGRA  0x80E1
#endif

static GLfloat vertices[] ={-40,40, -40,-40, 40,40, 40,-40}; 
static GLfloat texCoords[] = {0.0,1.0, 0.0,0.0, 1.0,1.0, 1.0,0.0};

float* mbOrthoMatrix(float m[16],float l,float r,float b,float t,float n,float f){
	m[0] = 2/(r-l);
	m[1] = 0;
	m[2] = 0;
	m[3] = 0;

	m[4] = 0;
	m[5] = 2/(t-b);
	m[6] = 0;
	m[7] = 0;

	m[8] = 0;
	m[9] = 0;
	m[10] = -2/(f-n);
	m[11] = 0;

	m[12] = (r+l)/(r-l);
	m[13] = (t+b)/(t-b);
	m[14] = (f+n)/(f-n);
	m[15] = 1;
	return m;
}

unsigned int createShader(char* shaderStr,unsigned int program,unsigned int shaderType){
	unsigned int shader;
	int	length, maxLength;
	int compiled  = GL_FALSE;
	char *infoLog;
	int sourceSize;

	sourceSize = (int)strlen(shaderStr);
	shader = glCreateShader(shaderType);

	glShaderSource(shader,1,(const char**)&shaderStr,&sourceSize); // Specify source code
	glCompileShader(shader); // Compile

	glGetShaderiv(shader,GL_COMPILE_STATUS,&compiled);
	glGetShaderiv(shader,GL_INFO_LOG_LENGTH,&maxLength);
	infoLog = new char[maxLength * sizeof(char)];
	glGetShaderInfoLog(shader,maxLength,&length,infoLog);

	// printf("%s\n",shaderStr);
	if(compiled){
		glAttachShader(program,shader); // Add
	}
	else{
		RDebug::Printf("%s",infoLog);
		RDebug::Printf("Compile failed");
	}
	delete [] infoLog;
	infoLog=0;
	return 0;
}

void linkShader(unsigned int program){
	int	length, maxLength;
	int linked = GL_FALSE;
	char *infoLog;

	// Optionally add more shaders to the program and link the program
	glLinkProgram(program);
	glGetProgramiv(program,GL_LINK_STATUS,&linked);

	glGetProgramiv(program,GL_INFO_LOG_LENGTH,&maxLength);
	infoLog = new char[maxLength * sizeof(char)];
	glGetProgramInfoLog(program,maxLength,&length,infoLog);

	// If all went well, make the program part of the current state
	if (!linked) {
		RDebug::Printf("%s",infoLog);
		RDebug::Printf("Link failed");
	}
	delete [] infoLog;
}

CTestTimer *CTestTimer::NewL(){
	CTestTimer* timer = new(ELeave)CTestTimer;
	CleanupStack::PushL(timer);
	timer->ConstructL();
	CleanupStack::Pop();
	return timer;
}

CTestTimer::CTestTimer() : CTimer(CActive::EPriorityIdle){
	CActiveScheduler::Add(this);
}

CTestTimer::~CTestTimer(){

}

void CTestTimer::Start(TTimeIntervalMicroSeconds32 aDelay,TTimeIntervalMicroSeconds32 anInterval,TCallBack aCallBack){
	iInterval=anInterval.Int();
	iCallBack=aCallBack;
	HighRes(aDelay);
}

void CTestTimer::RunL()
{
	HighRes(iInterval);
	iCallBack.CallBack();
}

CCubeView* CCubeView::NewL(const TRect& aRect){
	CCubeView* cmv = new(ELeave)CCubeView;
	CleanupStack::PushL(cmv);
	cmv->ConstructL(aRect);
	CleanupStack::Pop();
	return cmv;
}

CCubeView::~CCubeView(){
	delete iPeriodic;
	delete iBitmap;
	eglMakeCurrent(iDpy,0,0,0);
	eglDestroyContext(iDpy,iCtx);
	eglDestroySurface(iDpy,iDraw);
	eglTerminate(iDpy);
}

void CCubeView::ConstructL(const TRect& aRect){
	RDebug::Printf("CCubeView::ConstructL\n");
	TRAPD(error,CreateWindowL());
	if(error){
		// Something is wrong, but basically I just dislike warnings!
	}

	TRect rect(20,20,220,220);
	SetRect(rect);
	// SetExtentToWholeScreen();

	int minor = 0;
	int major = 0;
	EGLint numConfigs = 0;

	EGLConfig   cfg;
	EGLint context_attribs[] = {EGL_CONTEXT_CLIENT_VERSION, 2, EGL_NONE};

	EGLint attrib[] = {
		EGL_SURFACE_TYPE,EGL_WINDOW_BIT,
			EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
			EGL_RED_SIZE,8,
			EGL_GREEN_SIZE,8,
			EGL_BLUE_SIZE,8,
			EGL_ALPHA_SIZE,8,
			EGL_DEPTH_SIZE,16,
			EGL_NONE
	};

	iDpy = eglGetDisplay(EGL_DEFAULT_DISPLAY);

	if(!iDpy){
		RDebug::Printf("Failed to open the EGL default display");
	}

	if(!eglInitialize(iDpy,&major,&minor)){
		RDebug::Printf("Failed to init the EGL default display");
	}

	if(!eglChooseConfig(iDpy,attrib,&cfg,1,&numConfigs)||(numConfigs<1)){
		RDebug::Printf("Failed to find a config on the EGL default display");
	}

	iCtx = eglCreateContext(iDpy,cfg,EGL_NO_CONTEXT,context_attribs);
	if(!iCtx){
		RDebug::Printf("Failed to create a context");
	}

	// iDraw = eglCreateWindowSurface(iDpy,cfg,DrawableWindow(),0);
	iDraw = eglCreateWindowSurface(iDpy,cfg,&Window(),0);
	if(!iDraw){
		RDebug::Printf("Failed to create a context");
	}

	if(!eglMakeCurrent(iDpy,iDraw,iDraw,iCtx)){
		RDebug::Printf("Failed to make the context current");
	}

	m_program = glCreateProgram();
	unsigned int m_vertexShader = createShader(vertexShader,m_program,GL_VERTEX_SHADER);
	unsigned int m_fragmentShader = createShader(pixelShader,m_program,GL_FRAGMENT_SHADER);

	glBindAttribLocation(m_program, 0, "position");
	glBindAttribLocation(m_program, 1, "inTexCoord");

	linkShader(m_program);

	glUseProgram(m_program);

	glEnableVertexAttribArray (0);
	glEnableVertexAttribArray (1);

	glVertexAttribPointer(glGetAttribLocation(m_program,"position"), 2, GL_FLOAT, 0, 0, vertices);
	glVertexAttribPointer(glGetAttribLocation(m_program,"inTexCoord"), 2, GL_FLOAT, 0, 0, texCoords);

	glUniform1i(glGetUniformLocation(m_program,"mTexture"),1);
	glClearColor(0.0,0.0,1.0,0.0);

	unsigned int id;
	glGenTextures(1,&id);
	glBindTexture(GL_TEXTURE_2D,id);
	glPixelStorei(GL_UNPACK_ALIGNMENT,1);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_BGRA, 512, 511, 0, GL_BGRA, GL_UNSIGNED_BYTE, imageFighterRGBA512);

	GLenum e;
	do{
		e=glGetError();
		RDebug::Printf("main glGetError %u",e);
	}while(e!=GL_NO_ERROR);

	ActivateL();

	iRender = true;
	iPeriodic = CTestTimer::NewL();
	iPeriodic->Start(0,10,TCallBack(CCubeView::DrawCallback,this));
}

void CCubeView::draw(){
	float modelViewProj[16]; 
	int width=Rect().Width();
	int height=Rect().Height();

	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	glViewport(0,0,width,height);
	mbOrthoMatrix(modelViewProj,-width/10,width/10,-height/10,height/10,-100.0,100.0);
	glUniformMatrix4fv(glGetUniformLocation(m_program,"mWorldViewProj"),1,GL_FALSE,modelViewProj);
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	eglSwapInterval(iDpy,0);
	eglSwapBuffers(iDpy,iDraw);
}

TInt CCubeView::DrawCallback(TAny* aInstance){
	CCubeView* instance = (CCubeView*)aInstance;
	instance->draw();

	if(!(instance->iFrame%400)){
		User::ResetInactivityTime();
		TBuf<32> buf;
		instance->iEndTime.UniversalTime();
		buf.Format(_L("%f Hz"),400000000.0/(float)(instance->iEndTime.Int64()-instance->iBeginTime.Int64()));
		RDebug::Printf("Frame time: %f",400000000.0/(float)(instance->iEndTime.Int64()-instance->iBeginTime.Int64()));
		User::InfoPrint(buf);			
		instance->iBeginTime.UniversalTime();
	}
	instance->iFrame++;
	return 0;
}

void CCubeView::Draw(const TRect& aRect) const{
	RDebug::Printf("CRCTestView::Draw\n");
	/*
	TRgb color(0,128);
	CWindowGc& gc = SystemGc();
	gc.DrawRect(aRect);
	*/
}

TKeyResponse CCubeView::OfferKeyEventL(const TKeyEvent& aKeyEvent,TEventCode aType){
	TBuf<32> buf;

	if(aType == EEventKeyDown) {
		RDebug::Printf("Key code: %d",aKeyEvent.iScanCode);
		switch(aKeyEvent.iScanCode){
								case '0':
								case '1':
								case '2':
								case '3':
								case '4':
									SetRect(TRect(20,20,220,220));
									return EKeyWasConsumed;
								case '5':
									iTestCase=aKeyEvent.iScanCode-'0';
									return EKeyWasConsumed;
								case '6':
									iRender=iRender?false:true;
									return EKeyWasConsumed;
								case 'R':
									return EKeyWasConsumed;
								case 'S':
									SetExtentToWholeScreen();
									return EKeyWasConsumed;
								case 'T':
								case '7':
									SetExtentToWholeScreen();
									return EKeyWasConsumed;
								case 'Q':
								case 167:
								case 172:
								case 212:
								case 213:
								case '8':
									CBaActiveScheduler::Exit();
		}
	}
	return EKeyWasNotConsumed;
}

void CCubeView::HandlePointerEventL(const TPointerEvent& aPointerEvent){
	if(aPointerEvent.iType == TPointerEvent::EButton1Down)
	{
		CBaActiveScheduler::Exit();
	}
}


void CCubeAppUi::ConstructL(){

	BaseConstructL();
	iView=CCubeView::NewL(ClientRect());
	AddToStackL(iView);
}


CCubeAppUi::~CCubeAppUi(){
	RemoveFromStack(iView);
	delete iView;
}

void CCubeAppUi::HandleCommandL(TInt aCommand){
	switch (aCommand){
		// case EEikCmdGoBack:
		//	 CEikAppUi::Exit();
		//	 break;
	case EEikCmdExit: 
		CEikAppUi::Exit();
		break;
	default:
		break;
	}
}

void CCubeAppUi::HandleWsEventL(const TWsEvent& aEvent,CCoeControl* aDestination){
	RDebug::Printf("TWsEvent %d",aEvent.Type());
	switch (aEvent.Type())
	{
	case EEventFocusLost:
		// iView->Stop();
		break;
	case EEventFocusGained:
		// iView->Start();
		break;
	default:
		break;
	}
	CAknAppUi::HandleWsEventL(aEvent, aDestination);
}

CCubeDocument::CCubeDocument(CEikApplication& aApp) : CAknDocument(aApp){

}

CCubeDocument::~CCubeDocument(){

}

CCubeDocument* CCubeDocument::NewL(CEikApplication& aApp){
	CCubeDocument* self=new(ELeave) CCubeDocument(aApp);
	return self;
}

CAknAppUi* CCubeDocument::CreateAppUiL(){
	return(new(ELeave) CCubeAppUi);
}

TUid CCubeApplication::AppDllUid() const{
	return(KUidCube);
}

CApaDocument* CCubeApplication::CreateDocumentL(){
	return CCubeDocument::NewL(*this);
}
